#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <libraw1394/raw1394.h>

#define CSR_BASE        0xfffff0000000ULL

extern raw1394handle_t		g_handle;
extern nodeid_t			g_target;
extern int			g_target_ok;
extern u_int64_t		g_target_uid;
extern int			g_change_occurred;
extern int verbose,rem_ptrsize;

int verbose = 0;
int rem_ptrsize = sizeof(void *);
int auto_update;

raw1394handle_t		g_handle;
nodeid_t		g_target;
int			g_target_ok;
u_int64_t		g_target_uid;
int			g_change_occurred;
char*			g_sysmap;
size_t			g_sysmap_size;
int			g_machine;

static int
got_bus_reset(raw1394handle_t hndl, unsigned int generation)
{
	raw1394_update_generation(g_handle, generation);
	g_target_ok = 0;
	g_change_occurred = 1;
	fprintf(stderr, "Bus reset !\n");
	return 0;
}

static int
setup_handle(int port_num)
{
	int port_count;
	struct raw1394_portinfo ports[16];
	
	g_handle = raw1394_new_handle();
	if (!g_handle) {
		perror("libraw1394 couldn't be initialized");
		return -1;
	}

	port_count = raw1394_get_port_info(g_handle, ports, 16);
	if (port_count <= port_num) {
		fprintf(stderr, "Port %d not available (%d ports detected).\n",
			port_num, port_count);
		return -1;
	}
	raw1394_set_port(g_handle, port_num);
	fprintf(stderr, "Port %d (%s) opened, %d nodes detected\n", port_num,
		ports[port_num].name, ports[port_num].nodes);

	raw1394_set_bus_reset_handler(g_handle, got_bus_reset);

	return 0;
}

static void
select_target(void)
{
	int i, tgt, count, local;

	count = raw1394_get_nodecount(g_handle);
	local = raw1394_get_local_id(g_handle) & 0x3f;
	
	fprintf(stderr, "%d nodes available, local node is: %d\n", count, local);
start:
	for (i=0; i<count; i++) {
		quadlet_t uuid[2];
		int rc;

		fprintf(stderr, " %d: %04x, uuid: ", i, i | 0xffc0);
		rc = raw1394_read(g_handle, i | 0xffc0, CSR_BASE + 0x40c, 4, &uuid[0]);
		if (rc >= 0)
			rc = raw1394_read(g_handle, i | 0xffc0, CSR_BASE + 0x410, 4, &uuid[1]);
		if (rc < 0) {
			fprintf(stderr, "<err: %d>", errno);
			raw1394_reset_bus(g_handle);
			sleep(2);
			goto start;
		} else
			fprintf(stderr, "%08x %08x", uuid[0], uuid[1]);
		if (i == local)
			fprintf(stderr, " [LOCAL]");
		fprintf(stderr, "\n");
	}
	fprintf(stderr, "pick a target node: ");
	tgt = 0;
	if (tgt == local) {
		tgt++;
	}
	if (tgt < 0 || tgt >= raw1394_get_nodecount(g_handle)) {
		fprintf(stderr, "wrong node number !\n");
		exit(4);
        }
	else if (tgt == local) {
		fprintf(stderr, "can't pick local node !\n");
		exit(4);
	}
	g_target = tgt | 0xffc0;
	g_target_ok = 1;
	g_change_occurred = 1;
}

static void
status(void)
{
	fprintf(stderr, "Target : ");
	if (g_target_ok)
		fprintf(stderr, "%04x\n", g_target);
	else
		fprintf(stderr, "<unspecified>\n");

	fprintf(stderr, "Gen    : %d\n", raw1394_get_generation(g_handle));
}

int fwmem_read(unsigned long addr, void *data)
{
	if (!g_target_ok) {
		fprintf(stderr, "fwmem_read - no target...\n");
		return -2;
	}
	if (verbose)
		fprintf(stderr, "fwmem_read: addr %lx -> %p\n", addr, data);
	int rc = raw1394_read(g_handle, g_target,
			addr, 4, data);
	if (rc < 0) {
		perror("fwmem_read/raw1394_read");
		fprintf(stderr, "remote read failed, addr=%lx\n", addr);
	} else
		if (verbose)
			fprintf(stderr, "fwmem_read: SUCCESS: %lx\n", *(unsigned long *)data);
	return rc;
}

void* fwmem_readblk(void* dest, unsigned long addr, unsigned int len)
{
	unsigned int* cdst = (unsigned int*)dest;
	unsigned long end = addr + len;
	int startoffset = (unsigned long)addr & 0x3;

	addr -= startoffset;

	if (!g_target_ok) {
		fprintf(stderr, "readblk - no target...\n");
		return NULL;
	}
	while(addr < end) {
		if (fwmem_read(addr, cdst)) {
			fprintf(stderr, "readblk - aborting...\n");
			return NULL;
		}
		cdst++;
		addr += 4;
	}
	return dest + startoffset;
}

static int fwmem_writechar(unsigned long addr, char ch)
{
	fprintf(stderr, "fwmem_writechar( 0x%08lx << '%c' )\n", addr, ch);
	int rc = raw1394_write(g_handle, g_target, addr, 1, (quadlet_t *)&ch);
	if (rc < 0)
		perror("fwmem_writechar/raw1394_write");
	return rc;
}
int fwmem_writeblk(unsigned long addr, void *data, unsigned int len)
{
	char *dest = (char *)addr, *ptr = data;
	int rc = 0;
	fprintf(stderr, "fwmem_writeblk( 0x%08lx << 0x%p, %d )\n", addr, data, len);
	while (len-- > 0 &&
		!(rc = fwmem_writechar((unsigned long)dest++, *ptr++)));
	if (rc < 0)
		perror("fwmem_writeblk");
	return rc;
}

#define BLOCKSIZE 2048
//const int BLOCKSIZE=32;
char g_buf[BLOCKSIZE*2] __attribute__((__aligned__(BLOCKSIZE)));
char *g_data;
int g_continue =1, g_output, g_column;
unsigned long g_addr, g_towrite, g_end;
struct raw1394_reqhandle g_reqhandle[2];

void show_progress(unsigned long addr)
{
	fprintf(stderr, ".");
	g_column++;
	if (g_column >= 100) {
		g_column = 0;
		fprintf(stderr, " %4ld MB\n", addr/1024/1024);
	}
}

int my_start_next_read() {
	int result;

	int buf_num = ((g_addr % BLOCKSIZE)==0);
	g_reqhandle[buf_num].data = (void *)g_addr;

	struct raw1394_reqhandle *tag = &g_reqhandle[buf_num];

	//fprintf(stderr, "start read %d: %lx\n", buf_num, g_addr);
	result = raw1394_start_read(g_handle, g_target, g_addr, BLOCKSIZE,
						(quadlet_t *)g_data, (unsigned long)tag);
	if (result != 52)
		fprintf(stderr, "%lx: result: %d\n", g_addr, result);
	g_addr+=BLOCKSIZE;
	//g_data = g_buf[(!buf_num)*BLOCKSIZE];
	return (result != 52);
}

int my_callback(raw1394handle_t handle, void *data,
                              raw1394_errcode_t err)
{
	//fprintf(stderr, "callback: data=%p, err=%d\n", data, err);
	if (g_towrite != (unsigned long)data) {
		fprintf(stderr, "callback: towrite=%08lx\n", g_towrite);
		g_continue = 0;
	} else {
		//printf("buf:%s\n", g_data[((unsigned long)data % BLOCKSIZE)==0]);
		//int buf_num = ((g_addr % BLOCKSIZE)==0);
		//printf("buf %d:%s\n", buf_num, g_data);
		write(g_output, g_data, BLOCKSIZE);
		g_towrite = (unsigned long)data + BLOCKSIZE;
		my_start_next_read();
		if (g_addr >= g_end)
			return 1;
		if (!(g_addr % 200*1024*1024))
			show_progress(g_addr);
	}
	return 0;
}

int dump_memory(int fd, unsigned long addr, unsigned long end)
{
	int result = 0;
	g_reqhandle[0].callback = my_callback;
	g_reqhandle[1].callback = my_callback;
	g_towrite = addr;
	g_addr = addr;
	g_end  = end;
	g_data    = &g_buf[0];
	g_output = fd;

	fprintf(stderr, "len: %ld\n", end - addr);
	my_start_next_read();

	while (g_continue && !(result = raw1394_loop_iterate(g_handle)));

	if (result == 1)
		raw1394_loop_iterate(g_handle);
	else
		fprintf(stderr, "loop result: %d\n", result);
	if (result)
		perror("raw1394_loop_iterate");
	return result;
}

int main(int argc, char** argv)
{
	unsigned long addr = 0x64ee90;
	int fd;


	if (argc < 2) {
		fprintf(stderr, "Syntax: firedump <dump.bin>\n");
		//exit(5);
	}

	if (setup_handle(0) != 0)
		return 0;

	select_target();
	status();
	fd = 1;
	fprintf(stderr, "Writing dump to %s (fd %d)\n", argv[1], fd);
	//return dump_memory(fd, addr, addr+8192);
	return dump_memory(fd, 1024*1024, 2048*1024*1024UL);
	return dump_memory(fd, addr, 65536);

	if ((fd = open(argv[1], O_CREAT|O_EXCL|O_WRONLY, S_IRUSR|S_IWUSR)) == -1) {
		perror("open failed");
		fprintf(stderr, "firedump: could not create %s, aborting.\n", argv[1]);
		exit(6);
	}
	fprintf(stderr, "Exiting...\n");
	raw1394_destroy_handle(g_handle);
	free(g_sysmap);
	return 0;
}
