/****************************************************************************
 * GDB stub based on the stub files in gdb-6.4:
 *
 *    The following gdb commands are supported:
 *
 * command          function                               Return value
 *
 *    mAA..AA,LLLL  Read LLLL bytes at address AA..AA      hex data or ENN
 *    MAA..AA,LLLL: Write LLLL bytes at address AA.AA      OK or ENN
 *    XAA..AA,LLLL: Write LLLL binary bytes at address     OK or ENN
 *                  AA..AA
 *
 *    k             kill
 *
 *    ?             What was the last sigval ?             SNN   (signal NN)
 *                  -> Gives dummy answer at the moment
 *
 * All commands and responses are sent with a packet which includes a
 * checksum.  A packet consists of
 *
 * $<packet info>#<checksum>.
 *
 * where
 * <packet info> :: <characters representing the command or response>
 * <checksum>    :: <two hex digits computed as modulo 256 sum of <packetinfo>>
 *
 * When a packet is received, it is first acknowledged with either '+' or '-'.
 * '+' indicates a successful transfer.  '-' indicates a failed transfer.
 *
 * Example:
 *
 * Host:                  Reply:
 * $m0,10#2a               +$00010203040506070809101112131415#42
 *
 ****************************************************************************/

#include <stdio.h>
#include <string.h>
#include "fireproxy.h"

/*****************************************************************************
 * BUFMAX defines the maximum number of characters in inbound/outbound buffers
 * at least NUMREGBYTES*2 are needed for register packets 
 */

static const unsigned char hexchars[] = "0123456789abcdef";

#define NUMREGS 24

/* Number of bytes of registers.  */
#define NUMREGBYTES (NUMREGS * 4)

static char remcomTmpBuffer[BUFMAX];
static char remcomOutBuffer[BUFMAX];

/* Indicate to caller of mem2hex or hex2mem that there has been an error. */
static volatile int mem_err = 0;


/* Local functions:
 */
static unsigned char *hex2mem (unsigned char *, unsigned char *, int);
static unsigned char *bin2mem (unsigned char *, unsigned char *, int);
static char *mem2hex (unsigned char *, char *, int);

static void putpacket (char *);

static int hexToUlong (unsigned char **, unsigned long *);

/*
 * This function does all command procesing for interfacing to gdb.
 */

void handle_command (unsigned char *ptr)
{
      unsigned long addr, length;
	int binary;

      remcomOutBuffer[0] = 0;
      binary = 0;
      switch (*ptr++)
	{
	case 'R':
	  strcpy (remcomOutBuffer, "OK");
	  break;
	case '!':
	  strcpy (remcomOutBuffer, "OK");
	  break;
	case 'X': /* XAA..AA,LLLL:<binary data>#cs */
	  binary = 1;
	case 'M': /* MAA..AA,LLLL: Write LLLL bytes at address AA.AA return OK */
		if (hexToUlong (&ptr, &addr)
	            && *(ptr++) == ','
		    && hexToUlong (&ptr, &length)
		    && *(ptr++) == ':') {
		      if (binary)
			bin2mem(ptr, (unsigned char *)remcomTmpBuffer, length);
		      else
			hex2mem(ptr, (unsigned char *)remcomTmpBuffer, length);
			  if (fwmem_writeblk(virt_to_phys(addr),
						remcomTmpBuffer,
						length))
				strcpy (remcomOutBuffer, "E03");
			  else
			  	strcpy (remcomOutBuffer, "OK");
		      break;
		}
		strcpy (remcomOutBuffer, "E02");
		break;
	case 'm': /* mAA..AA,LLLL  Read LLLL bytes at address AA..AA */
	  if (hexToUlong (&ptr, &addr)
		&& *(ptr++) == ','
	     	&& hexToUlong (&ptr, &length)
		&& length < BUFMAX/2) {
		  void *ret = fwmem_readblk(remcomTmpBuffer,
				virt_to_phys(addr), length);
		  if (ret)
			mem2hex (ret, remcomOutBuffer, length);
		  else
		  	strcpy (remcomOutBuffer, "E03");
	  } else
		  strcpy (remcomOutBuffer, "E01");
	  break;
	case '?': // Dummy answer:
	  remcomOutBuffer[0] = 'S';
	  remcomOutBuffer[1] = '0';
	  remcomOutBuffer[2] = '0';
	  remcomOutBuffer[3] = 0;
	  break;
	case 'G':	/* set the value of the CPU registers - return OK */
	  // hex2mem (ptr, (unsigned char *) registers, NUMREGBYTES, 0);
	case 'D':		/* Detach */
	  strcpy (remcomOutBuffer, "OK");
	  break;
	case 'g':		/* return the value of the CPU registers */
	case 'p':		/* get register */
	  strcpy (remcomOutBuffer, "0000");
	  break;
	case 'k':		/* kill the program */
	  fprintf(logfile, "got kill command: '%c'\n", *(ptr-1));
	  strcpy (remcomOutBuffer, "OK");
	default:	/* Unknown code.  Return an empty reply message. */
	  break;
	}			/* switch */

      /* reply to the request */
      putpacket (remcomOutBuffer);
}

static int hex (unsigned char ch)
{
  if ((ch >= 'a') && (ch <= 'f'))
    return (ch - 'a' + 10);
  if ((ch >= '0') && (ch <= '9'))
    return (ch - '0');
  if ((ch >= 'A') && (ch <= 'F'))
    return (ch - 'A' + 10);
  return (-1);
}

/* Convert the hex array pointed to by buf into binary to be placed in mem.
   Return a pointer to the character AFTER the last byte written. */

static unsigned char *
hex2mem (unsigned char *buf, unsigned char *mem, int count)
{
  int i;
  unsigned char ch;

  for (i = 0; i < count; i++)
    {
      ch = hex (*buf++) << 4;
      ch = ch + hex (*buf++);
      *mem++ = ch;
    }
  return (mem);
}

/* Convert the binary stream in BUF to memory.

   Gdb will escape $, #, and the escape char (0x7d).
   COUNT is the total number of bytes to write into
   memory. */
static unsigned char *
bin2mem (unsigned char *buf, unsigned char *mem, int count)
{
  int i;

  for (i = 0; i < count; i++)
    {
      /* Check for any escaped characters. Be paranoid and
         only unescape chars that should be escaped. */
      if (*buf == 0x7d)
	{
	  switch (*(buf + 1))
	    {
	    case 0x3:		/* # */
	    case 0x4:		/* $ */
	    case 0x5d:		/* escape char */
	      buf++;
	      *buf |= 0x20;
	      break;
	    default:
	      /* nothing */
	      break;
	    }
	}
      *mem++ = *buf++;
    }

  return mem;
}

static void
putpacket (char *buffer)
{
  unsigned char checksum;
  int count;
  char ch;

 //fprintf(logfile, "answering with  : %s\n", *buffer ? buffer : "<command not supported>");
  /*  $<packet info>#<checksum>. */
      putDebugChar ('$');
      checksum = 0;
      count = 0;

      while ((ch = buffer[count]))
	{
	  putDebugChar (ch);
	  checksum += ch;
	  count += 1;
	}
      putDebugChar ('#');
      putDebugChar (hexchars[checksum >> 4]);
      putDebugChar (hexchars[checksum % 16]);
	fflush(outfile);
}

/* Convert the memory pointed to by mem into hex, placing result in buf.
   Return a pointer to the last char put in buf (null).
   If MAY_FAULT is non-zero, then we should set mem_err in response to
   a fault; if zero treat a fault like any other fault in the stub.
*/

static char *
mem2hex (unsigned char *mem, char *buf, int count)
{
  int i;
  unsigned char ch;

  for (i = 0; i < count; i++) {
      ch = *mem++;
      *buf++ = hexchars[ch >> 4];
      *buf++ = hexchars[ch % 16];
  }
  *buf = 0;
  return (buf);
}

/**********************************************/
/* WHILE WE FIND NICE HEX CHARS, BUILD AN INT */
/* RETURN NUMBER OF CHARS PROCESSED           */
/**********************************************/
static int
hexToUlong (unsigned char **ptr, unsigned long *intValue)
{
  int numChars = 0;
  int hexValue;

  *intValue = 0;
  while (**ptr)
    {
      hexValue = hex (**ptr);
      if (hexValue >= 0)
	{
	  *intValue = (*intValue << 4) | hexValue;
	  numChars++;
	}
      else
	break;
      (*ptr)++;
    }
  return (numChars);
}
