/*************************************************************************
 *
 *  $RCSfile: mbxformt.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: mhu $ $Date: 2001/07/24 19:39:12 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _CHAOS_MBXFORMT_HXX
#define _CHAOS_MBXFORMT_HXX

#ifndef _ERRCODE_HXX
#include <tools/errcode.hxx>
#endif

#ifndef _CNTVWITM_HXX
#include <cntvwitm.hxx>
#endif

class DateTime;

namespace chaos
{

class CntItemListItem;
class CntRangesItem;

//============================================================================
class CntMBXChars
{
public:
	/** Character classifications.

		@descr  Each character of type 'unsigned char' belongs to exactly one
		class.

		@descr  There are three different <CRLF> concepts:
		- 'Pair CRLF' means <CRLF> is a character of class PRCR followed by a
		  character of class PRLF.
		- 'Optional CRLF' means <CRLF> is any either a character of class
		  POCR, optionally followd by a character of class POLF, or only a
		  character of class POLF.
		- 'Single CRLF' means <CRLF> is a character of class SNCR.

		@descr  The examples given are for 'classical' RFC 822.
	 */
	enum Class
	{
		LWSP, /// linear white space ('\t', ' ')
		SPCL, /// special (',', '.', ':', etc.)
		ATOM, /// atom ('a', 'b', 'c', etc.)
		QSTR, /// quoted string open/close ('"')
		DOPN, /// domain literal open ('[')
		DCLS, /// domain literal close (']')
		COPN, /// comment open ('(')
		CCLS, /// comment close (')')
		ESCP, /// quoted pair escape ('\\')
		PRCR, /// CR part of 'pair CRLF' ('\r')
		PRLF, /// LF part of 'pair CRLF' ('\n')
		POCR, /// CR part of 'optional CRLF'
		POLF, /// LF part of 'optional CRLF'
		SNCR, /// CR that is 'single CRLF'
		CTLZ, /// ignored if last char of input
		ZERO, /// like end of input
		OTHR /// all other ('\x00', '\x01', \x02', etc.)
	};

private:
	sal_uInt8 const * m_pData;

public:
	CntMBXChars(sal_uInt8 const * pTheData): m_pData(pTheData) {}

	operator sal_uInt8 const *() const { return m_pData; }

	Class getClass(sal_Char cChar) const
	{ return Class(m_pData[sal_uChar(cChar)]); }

	// 'Enhanced' RFC 822:
	//
	//  LWSP: '\t', ' '
	//  SPCL: ',', '.', ':', ';', '<', '>', '@'
	//  ATOM: '!', '#', '$', '%', '&', '\'', '*', '+', '-', '/', '0'--'9',
	//        '=', '?', 'A'--'Z', '^', '_', '`', 'a'--'z', '{', '|', '}', '~',
	//        '\x80'--'\xFF'
	//  QSTR: '"'
	//  DOPN: '['
	//  DCLS: ']'
	//  COPE: '('
	//  CCLS: ')'
	//  ESCP: '\\'
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsRFC822[256];

	// 'Enhanced' MIME:
	//
	//  LWSP: '\t', ' '
	//  SPCL: ',', '/', ':', ';', '<', '=', '>', '?', '@',
	//  ATOM: '!', '#', '$', '%', '&', '\'', '*', '+', '-', '.', '0'--'9',
	//        'A'--'Z', '^', '_', '`', 'a'--'z', '{', '|', '}', '~', '\x80'--
	//        '\xFF'
	//  QSTR: '"'
	//  DOPN: '['
	//  DCLS: ']'
	//  COPE: '('
	//  CCLS: ')'
	//  ESCP: '\\'
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsMIME[256];

	// Special character classifications for boundaries:
	//
	//  LWSP: '\t', ' '
	//  SPCL:
	//  ATOM: '\'', '(', ')', '+', ',', '-', '.' '/', '0'--'9', ':', '=', '?',
	//        'A'--'Z', '_', 'a'--'z', '\x80'--'\xFF'
	//  QSTR:
	//  DOPN:
	//  DCLS:
	//  COPE:
	//  CCLS:
	//  ESCP:
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsBoundary[256];

	// Special character classifications for message bodies:
	//
	//  LWSP:
	//  SPCL:
	//  ATOM: '\x00'--'\x09', '\x0B'--'\x0C', '\x0E'--'\x19', '\x1B'--'\x1F',
	//        ' ', '!', '"', '#', '$', '%', '&', '\'', '(', ')', '*', '+',
	//        ',', '-', '.', '/', '0'--'9', ':', ';', '<', '=', '>', '?', '@',
	//        'A'--'Z', '[', '\\', ']', '^', '_', '`', 'a'--'z', '{', '|',
	//        '}', '~', '\x7F'--'\xFF'
	//  QSTR:
	//  DOPN:
	//  DCLS:
	//  COPE:
	//  CCLS:
	//  ESCP:
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsBody[256];

	// Special character classifications for numeric ranges:
	//
	//  LWSP: '\t', ' '
	//  SPCL: ',', '-'
	//  ATOM: '0'--'9'
	//  QSTR:
	//  DOPN:
	//  DCLS:
	//  COPE: '('
	//  CCLS: ')'
	//  ESCP: '\\'
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsRanges[256];

	// Special character classifications for 'simple text':
	//
	//  LWSP: '\t' ' '
	//  SPCL:
	//  ATOM: '!', '"', '#', '$', '%', '&', '\'', '(', ')', '*', '+', ',',
	//        '-', '.', '/', '0'--'9', ':', ';', '<', '=', '>', '?', '@',
	//        'A'--'Z', '[', '\\', ']', '^', '_', '`', 'a'--'z', '{', '|',
	//        '}', '~', '\x80'--'\xFF'
	//  QSTR:
	//  DOPN:
	//  DCLS:
	//  COPE:
	//  CCLS:
	//  ESCP:
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsText[256];

	// 'Enhanced' RFCs 822, 2047 *text:
	//
	//  LWSP: '\t', ' '
	//  SPCL: '=', '?'
	//  ATOM: '\x00'--'\x08', '\x0B'--'\x0C', '\x0E'--'\x19', '\x1B'--'\x1F',
	//        '"', '!', '#', '$', '%', '&', '\'', '(', ')', '*', '+', ',',
	//        '-', '.', '/', '0'--'9', ':', ';', '<', '>', '@', 'A'--'Z', '[',
	//        '\\', ']', '^', '_', '`', 'a'--'z', '{', '|', '}', '~', '\x7F'--
	//        '\xFF'
	//  QSTR:
	//  DOPN:
	//  DCLS:
	//  COPE:
	//  CCLS:
	//  ESCP:
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsSText[256];

	// 'Enhanced' RFC 2047 encoded word token:
	//
	//  LWSP: '\t', ' '
	//  SPCL: '=', '?'
	//  ATOM: '!', '#', '$', '%', '&', '\'', '*', '+', '-', '0'--'9', 'A'--
	//        'Z', '^', '_', '`', 'a'--'z', '{', '|', '}', '~'
	//  QSTR:
	//  DOPN:
	//  DCLS:
	//  COPE:
	//  CCLS:
	//  ESCP:
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsEToken[256];

	// 'Enhanced' RFC 2047 encoded word encoded-text:
	//
	//  LWSP: '\t', ' '
	//  SPCL: '?'
	//  ATOM: '"', '!', '#', '$', '%', '&', '\'', '(', ')', '*', '+', ',',
	//        '-', '.', '/', '0'--'9', ':', ';', '<', '=', '>', '@', 'A'--'Z',
	//        '[', '\\', ']', '^', '_', '`', 'a'--'z', '{', '|', '}', '~'
	//  QSTR:
	//  DOPN:
	//  DCLS:
	//  COPE:
	//  CCLS:
	//  ESCP:
	//  PRCR:
	//  PRLF:
	//  POCR: '\r'
	//  POLF: '\n'
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsEText[256];

	// The above classifications, each with '\x00' classified as ZERO:
	static sal_uInt8 const aCharsRFC822NUL[256];
	static sal_uInt8 const aCharsMIMENUL[256];
	static sal_uInt8 const aCharsBoundaryNUL[256];
	static sal_uInt8 const aCharsBodyNUL[256];
	static sal_uInt8 const aCharsRangesNUL[256];
	static sal_uInt8 const aCharsTextNUL[256];
	static sal_uInt8 const aCharsSTextNUL[256];
	static sal_uInt8 const aCharsETokenNUL[256];
	static sal_uInt8 const aCharsETextNUL[256];

	// Special character classifications for plain, single characters:
	//
	//  LWSP:
	//  SPCL:
	//  ATOM:
	//  QSTR:
	//  DOPN:
	//  DCLS:
	//  COPE:
	//  CCLS:
	//  ESCP:
	//  PRCR:
	//  PRLF:
	//  POCR:
	//  POLF:
	//  SNCR:
	//  CTLZ: '\x1A'
	//  ZERO:
	static sal_uInt8 const aCharsPlain[256];
};

//============================================================================
class CntMBXAtom
{
	friend class CntMBXAtomTable;

	ByteString m_aName;
	ByteString m_aCanonic;
	CntMBXAtom * m_pLeft;
	CntMBXAtom * m_pRight;
	sal_uInt32 m_nKey;

	inline CntMBXAtom(ByteString const & rTheName,
					  ByteString const & rTheCanonic, sal_uInt32 nTheKey);

	~CntMBXAtom();

public:
	ByteString getName() const { return m_aName; }

	ByteString getCanonic() const { return m_aCanonic; }

	sal_uInt32 getKey() const { return m_nKey; }
};

inline CntMBXAtom::CntMBXAtom(ByteString const & rTheName,
							  ByteString const & rTheCanonic,
							  sal_uInt32 nTheKey):
	m_aName(rTheName),
	m_aCanonic(rTheCanonic),
	m_nKey(nTheKey),
	m_pLeft(0),
	m_pRight(0)
{}

//============================================================================
class CntMBXAtomTable
{
public:
	enum
	{
		ATOM_BASE64,
		ATOM_BCC,
		ATOM_CC,
		ATOM_CHAOS,
		ATOM_COMMENTS,
		ATOM_COMPLETELY_LOCALLY_SENT,
		ATOM_CONFIRMED,
		ATOM_CONTENT_BASE,
		ATOM_CONTENT_DESCRIPTION,
		ATOM_CONTENT_DISPOSITION,
		ATOM_CONTENT_ID,
		ATOM_CONTENT_LOCATION,
		ATOM_CONTENT_TRANSFER_ENCODING,
		ATOM_CONTENT_TYPE,
		ATOM_COPY,
		ATOM_DATE,
		ATOM_EXTERNAL_ERROR,
		ATOM_FROM,
		ATOM_GROUP,
		ATOM_HIDDEN,
		ATOM_IN_REPLY_TO,
		ATOM_KEYWORDS,
		ATOM_MAPI,
		ATOM_MARKED,
		ATOM_MBOX,
		ATOM_MESSAGE,
		ATOM_MESSAGE_ID,
		ATOM_MESSAGEBOX,
		ATOM_MIME_VERSION,
		ATOM_NEWS,
		ATOM_NEWSBOX,
		ATOM_NEWSGROUPS,
		ATOM_NNTP,
		ATOM_NO,
		ATOM_NONRECOVERABLE_LOCAL_ERROR,
		ATOM_PARTIALLY_LOCALLY_SENT,
		ATOM_PASSWORD,
		ATOM_PROTOCOL,
		ATOM_QUOTED_PRINTABLE,
		ATOM_READ,
		ATOM_RECEIVED,
		ATOM_RECOVERABLE_LOCAL_ERROR,
		ATOM_REFERENCES,
		ATOM_REPLY,
		ATOM_REPLY_CODE,
		ATOM_REPLY_TO,
		ATOM_RETURN_PATH,
		ATOM_SENDER,
		ATOM_SERVER,
		ATOM_SMTP,
		ATOM_STATE,
		ATOM_SUBJECT,
		ATOM_SUBSCRIBED,
		ATOM_TO,
		ATOM_TRIES,
		ATOM_USER,
		ATOM_VIM,
		ATOM_WAITING_CONFIRMATION,
		ATOM_WRITTEN,
		ATOM_X_CHAOS_MARKED,
		ATOM_X_CHAOS_READ,
		ATOM_X_CHAOS_RECIPIENTS,
		ATOM_X_CHAOS_SIZE,
		ATOM_X_MAILER,
		ATOM_X_MOZILLA_STATUS,
		ATOM_X_PRIORITY,
		ATOM_XREF,
		ATOM_YES,
		DYNAMIC_KEY = 0x80000000
	};

private:
	CntMBXAtom * m_pRoot;

	CntMBXAtom const & insert(ByteString const & rName,
							  ByteString const & rCanonic, sal_uInt32 nKey);

public:
	CntMBXAtomTable();

	~CntMBXAtomTable() { delete m_pRoot; }

	CntMBXAtom const & insert(ByteString const & rName,
							  ByteString const & rCanonic)
	{ return insert(rName, rCanonic, DYNAMIC_KEY); }
};

//============================================================================
class CntMBXToken
{
public:
	// The rules for the token types:
	//
	//  TYPE_LWSP = 1*([TYPE_CRLF] <LWSP>)
	//
	//  TYPE_SPECIAL = <SPCL> / <DCLS> / <CCLS> / <ESCP>
	//
	//  TYPE_ATOM = 1*<ATOM>
	//
	//  TYPE_QSTR = <QSTR> *(qtext / quoted-pair) <QSTR>
	//
	//  TYPE_BROKEN_QSTR = <QSTR> *(qtext / quoted-pair)
	//                      / <QSTR> *(qtext / quoted-pair) illegal
	//                         *(qtext / quoted-pair / illegal) [<QSTR>]
	//
	//  TYPE_DOMAINLIT = <DOPN> *(dtext / quoted-pair) <DCLS>
	//
	//  TYPE_BROKEN_DOMAINLIT = <DOPN> *(dtext / quoted-pair)
	//                           / <DOPN> *(dtext / quoted-pair)
	//                              (<DOPN> / illegal)
	//                              *(dtext / quoted-pair / <DOPN> / illegal)
	//                              [<DCLS>]
	//
	//  TYPE_COMMENT = <COPN> *(ctext / quoted-pair / TYPE_COMMENT) <CCLS>
	//
	//  TYPE_BROKEN_COMMENT = open-comment / illegal-comment
	//
	//  TYPE_CRLF = (<PRCR> <PRLF>) / (<POCR> [<POLF>]) / <POLF> / <SNCR>
	//
	//  TYPE_OTHER = <OTHR> / <PRCR> / <PRLF> / (<CTLZ> <<~EOF>>)
	//
	//  TYPE_ZERO = ZERO  ; the ZERO is not consumed!
	//
	//  TYPE_END = [<CTLZ>] <<EOF>>
	//
	// The auxiliary rules used:
	//
	//  qtext = <LWSP> / <SPCL> / <ATOM> / <DOPN> / <DCLS> / <COPN> / <CCLS>
	//           / <PRLF> / <OTHR> / (<CTLZ> <<~EOF>>) / (TYPE_CRLF <LWSP>)
	//
	//  dtext = <LWSP> / <SPCL> / <ATOM> / <QSTR> / <COPN> / <CCLS> / <PRLF>
	//           / <OTHR> / (<CTLZ> <<~EOF>>) / (TYPE_CRLF <LWSP>)
	//
	//  ctext = <LWSP> / <SPCL> / <ATOM> / <QSTR> / <DOPN> / <DCLS> / <PRLF>
	//           / <OTHR> / (<CTLZ> <<~EOF>>) / (TYPE_CRLF <LWSP>)
	//
	//  open-comment = <COPN>
	//                  *(ctext / quoted-pair / TYPE_COMMENT / illegal
	//                     / illegal-comment)
	//                  [open-comment]
	//
	//  illegal-comment = <COPN> *(ctext / quoted-pair / TYPE_COMMENT)
	//                     (illegal / illegal-comment)
	//                     *(ctext / quoted-pair / TYPE_COMMENT / illegal
	//                        / illegal-comment)
	//                     <CCLS>
	//
	//  quoted-pair = <ESCP>
	//                 (<LWSP> / <SPCL> / <ATOM> / <QSTR> / <DOPN> / <DCLS>
	//                   / <COPN> / <CCLS> / <ESCP> / <PRCR> / <PRLF> / <POCR>
	//                   / <POLF> / <SNCR> / <OTHR> / (<CTLZ> <<~EOF>>))
	//
	//  illegal = <PRCR> / (<ESCP> [<CTLZ>] <<EOF>>)
	//
	// The special rule 'r <<EOF>>' means rule r at end of file, the special
	// rule 'r <<~EOF>>' means rule r not at end of file.
	enum Type
	{
		TYPE_LWSP,
		TYPE_SPECIAL,
		TYPE_ATOM,
		TYPE_QSTR,
		TYPE_BROKEN_QSTR,
		TYPE_DOMAINLIT,
		TYPE_BROKEN_DOMAINLIT,
		TYPE_COMMENT,
		TYPE_BROKEN_COMMENT,
		TYPE_CRLF,
		TYPE_OTHER,
		TYPE_ZERO,
		TYPE_END
	};

private:
	ByteString m_aValue;
	ByteString m_aCanonic;
	CntMBXAtom const * m_pAtom;
	Type m_eType;

public:
	CntMBXToken(): m_eType(TYPE_END), m_pAtom(0) {}

	inline CntMBXToken(Type eTheType, ByteString const & rTheValue);

	inline CntMBXToken(Type eTheType, sal_Char cTheValue);
		// Due to an inconsistency in the old String(unsigned char) ctor,
		// correct creation of, e.g., CntMBXToken(TYPE_CRLF, *p) with
		// unsigned char * p pointing to '\0' was not possible without this
		// extra ctor.  Is this still necessary? @@@

	inline CntMBXToken(Type eTheType, ByteString const & rTheValue,
					   ByteString const & rTheCanonic,
					   CntMBXAtom const * pTheAtom = 0);

	Type getType() const { return m_eType; }

	ByteString const & getValue() const { return m_aValue; }

	inline sal_Char getCharValue() const;

	ByteString const & getCanonic() const { return m_aCanonic; }

	CntMBXAtom const * getAtom() const { return m_pAtom; }
};

inline CntMBXToken::CntMBXToken(Type eTheType, ByteString const & rTheValue):
	m_eType(eTheType),
	m_aValue(rTheValue),
	m_aCanonic(rTheValue),
	m_pAtom(0)
{}

inline CntMBXToken::CntMBXToken(Type eTheType, ByteString const & rTheValue,
								ByteString const & rTheCanonic,
								CntMBXAtom const * pTheAtom):
	m_eType(eTheType),
	m_aValue(rTheValue),
	m_aCanonic(rTheCanonic),
	m_pAtom(pTheAtom)
{}

inline CntMBXToken::CntMBXToken(Type eTheType, sal_Char cTheValue):
	m_eType(eTheType),
	m_pAtom(0)
{
	m_aValue = ByteString(&cTheValue, 1);
	m_aCanonic = m_aValue;
}

inline sal_Char CntMBXToken::getCharValue() const
{
	DBG_ASSERT(m_aValue.Len() == 1,
			   "CntMBXToken::getCharValue() w/ m_aValue.Len() != 1");
	return m_aValue.GetChar(0);
}

//============================================================================
class CntMBXScanner
{
public:
	enum SourceFormat
	{
		FORMAT_GENERAL,
		FORMAT_OUTLOOK_EXPRESS
	};

	// CANONIC_LWSP sets a TYPE_LWSP token's canonic value to the respective
	//  token value with foldings unfolded (the leading TYPE_CRLF token of
	//  each folding removed).
	//
	// CANONIC_ATOM sets a TYPE_ATOM token's canonic value to the lower case
	//  equivalent of the respective token value.
	//
	// CANONIC_QSTR sets a TYPE_QSTR or TYPE_BROKEN_QSTR token's canonic value
	//  to the respective token value with surrounding QSTR characters
	//  removed, quoted pairs resolved (the leading ESCP character of each
	//  quoted pair removed), and foldings unfolded.
	//
	// CANONIC_DOMAINLIT sets a TYPE_DOMAINLIT or TYPE_BROKEN_DOMAINLIT
	//  token's canonic value to the respective token value with surrounding
	//  DOPN/DCLS characters removed, quoted pairs resolved, and foldings
	//  unfolded.
	enum ScanMode
	{
		FOLDED_LWSP = 0x01,
		CANONIC_LWSP = 0x02,
		CANONIC_ATOM = 0x04,
		CANONIC_QSTR = 0x08,
		CANONIC_DOMAINLIT = 0x10,
		HASH_ATOM = 0x20,
		SKIP_LWSP = 0x40,
		SKIP_COMMENTS = 0x80,
		NORMAL = FOLDED_LWSP | CANONIC_ATOM | CANONIC_QSTR | CANONIC_DOMAINLIT
		             | HASH_ATOM | SKIP_LWSP | SKIP_COMMENTS
	};

	enum ScanBodyMode
	{
		SCAN_BODY_NO,
		SCAN_BODY_INIT,
		SCAN_BODY_YES
	};

private:
	enum { MAX_BACK_UP = 2 };

	enum Class { CLASS_END = CntMBXChars::OTHR + 1 };

	CntMBXAtomTable const * m_pSavedTokenAtomTable[MAX_BACK_UP];
	sal_uInt8 const * m_pSavedTokenChars[MAX_BACK_UP];
	ScanMode m_eSavedTokenMode[MAX_BACK_UP];
	CntMBXToken m_aSavedToken[MAX_BACK_UP];
	ULONG m_nSavedTokenStart[MAX_BACK_UP];
	sal_Char const * m_pBufferPointer;
	sal_Char const * m_pBufferEnd;
	sal_Char const * m_pSavedBufferEnd;
	ULONG m_nBufferStart;
	ULONG m_nScanBodyEnd;
	int m_nSavedTokenCount;
	int m_nBackedTokenCount;
	int m_nStackBottom;
	SourceFormat m_eSourceFormat;
	bool m_bSourceFormatDetermined;

	inline Class getClass(CntMBXChars const & rChars) const;

	inline ErrCode readToken(ULONG nStart, ByteString & rToken);

	inline ErrCode seekBack(ULONG nPos);

	ErrCode checkBodyEnd(ULONG nOffset);

	void setBodyEnd();

	inline void setScanBodyBufferEnd();

protected:
	sal_Char * m_pBuffer;

	inline void init();
	 // Due to a bug in gcc, derived classes can't call ctor of CntMBXScanner,
	 // so this function is needed.

	void deinit();
	 // This function must be called from each derived dtor! It calls
	 // removeMark() and thus can't be called in base dtor...

	virtual ErrCode readBuffer(ULONG nStart, ULONG & rLen) = 0;

	virtual ErrCode readBlock(ULONG nStart, xub_StrLen nLen,
							  ByteString & rBlock, bool bPeek) = 0;

	virtual void addMark(ULONG nPos) = 0;

	virtual void removeMark(ULONG nPos) = 0;

	virtual ErrCode seek(ULONG nPos, ULONG & rBufferStart, ULONG & rBufferLen)
		= 0;

public:
	virtual ~CntMBXScanner() {}

	ErrCode determineSourceFormat(SourceFormat & rFormat);

	ErrCode scan(CntMBXToken & rToken, CntMBXAtomTable & rTable,
				 sal_uInt8 const * pChars = CntMBXChars::aCharsRFC822,
				 ScanMode eMode = NORMAL,
				 ScanBodyMode eScanBody = SCAN_BODY_NO);

	bool backUp();

	bool backUpNoEnd();

	bool peekSaved(unsigned nIndex, CntMBXToken const ** pToken = 0,
				   CntMBXAtomTable const ** pAtomTable = 0,
				   sal_uInt8 const ** pChars = 0,
				   ScanMode const ** pMode = 0);

	virtual ULONG getInputSize() = 0;

	ULONG getInputProcessed() const
	{ return m_nBufferStart + (m_pBufferPointer - m_pBuffer); }

	inline void scanBodyDone();
};

inline
CntMBXScanner::Class CntMBXScanner::getClass(CntMBXChars const & rChars) const
{
	return m_pBufferPointer < m_pBufferEnd ?
		       Class(rChars.getClass(*m_pBufferPointer)) : CLASS_END;
}

inline ErrCode CntMBXScanner::readToken(ULONG nStart, ByteString & rToken)
{
	return readBlock(nStart,
					 xub_StrLen(m_nBufferStart
								    + (m_pBufferPointer - m_pBuffer)
								    - nStart),
					 rToken, false);
}

inline void CntMBXScanner::setScanBodyBufferEnd()
{
	if (m_nScanBodyEnd < m_nBufferStart)
	{
		m_pSavedBufferEnd = m_pBufferEnd;
		m_pBufferEnd = m_pBuffer;
	}
	else if (m_nScanBodyEnd - m_nBufferStart
			     < ULONG(m_pBufferEnd - m_pBuffer))
	{
		m_pSavedBufferEnd = m_pBufferEnd;
		m_pBufferEnd = m_pBuffer + (m_nScanBodyEnd - m_nBufferStart);
	}
}

inline void CntMBXScanner::init()
{
	m_nBufferStart = 0;
	m_pBufferPointer = 0;
	m_pBufferEnd = 0;
	m_bSourceFormatDetermined = false;
	m_eSourceFormat = FORMAT_GENERAL;
	m_nSavedTokenCount = 0;
	m_nBackedTokenCount = 0;
	m_nStackBottom = 0;
	m_nScanBodyEnd = ULONG(-1);
	m_pSavedBufferEnd = 0;
	m_pBuffer = 0;
}

inline void CntMBXScanner::scanBodyDone()
{
	if (m_pSavedBufferEnd)
	{
		m_pBufferEnd = m_pSavedBufferEnd;
		m_pSavedBufferEnd = 0;
	}
	m_nScanBodyEnd = ULONG(-1);
}

//============================================================================
class CntMBXStringScanner: public CntMBXScanner
{
	ByteString m_aString;

protected:
	virtual ErrCode readBuffer(ULONG nStart, ULONG & rLen);

	virtual ErrCode readBlock(ULONG nStart, xub_StrLen nLen,
							  ByteString & rBlock, bool);

	virtual void addMark(ULONG) {}

	virtual void removeMark(ULONG) {}

	virtual ErrCode seek(ULONG nPos, ULONG & rBufferStart,
						 ULONG & rBufferLen);

public:
	inline CntMBXStringScanner(ByteString const & rTheString);

	virtual ~CntMBXStringScanner();

	virtual ULONG getInputSize() { return m_aString.Len(); }
};

inline CntMBXStringScanner::CntMBXStringScanner(ByteString const &
												    rTheString):
	m_aString(rTheString)
{
	m_aString.GetBufferAccess();
	init();
}

//============================================================================
class CntMBXStreamScanner: public CntMBXScanner
{
	enum { BUFFER_SIZE = 8192 };

	SvStream & m_rStream;
	ULONG m_nOffset;
	ULONG m_nStreamBufferStart;
	ULONG m_nStreamBufferLen;

protected:
	virtual ErrCode readBuffer(ULONG nStart, ULONG & rLen);

	virtual ErrCode readBlock(ULONG nStart, xub_StrLen nLen,
							  ByteString & rBlock, bool bPeek);

	virtual void addMark(ULONG nPos);

	virtual void removeMark(ULONG nPos);

	virtual ErrCode seek(ULONG nPos, ULONG & rBufferStart,
						 ULONG & rBufferLen);

public:
	CntMBXStreamScanner(SvStream & rTheStream);

	virtual ~CntMBXStreamScanner();

	virtual ULONG getInputSize();
};

//============================================================================
class CntMBXBodyScanner
{
	CntMBXScanner * m_pScanner;
	CntMBXAtomTable & m_rAtomTable;
	ULONG m_nCount;

public:
	CntMBXBodyScanner(CntMBXScanner * pTheScanner,
					  CntMBXAtomTable & rTheAtomTable):
		m_pScanner(pTheScanner), m_rAtomTable(rTheAtomTable), m_nCount(0) {}

	inline ErrCode scan(CntMBXToken & rToken,
						sal_uInt8 const * pChars = CntMBXChars::aCharsRFC822,
						CntMBXScanner::ScanMode eMode
						    = CntMBXScanner::NORMAL);

	inline bool backUp();

	ULONG getInputProcessed() const
	{ return m_pScanner->getInputProcessed(); }
};

inline ErrCode CntMBXBodyScanner::scan(CntMBXToken & rToken,
									   sal_uInt8 const * pChars,
									   CntMBXScanner::ScanMode eMode)
{
	ErrCode nError = m_pScanner->scan(rToken, m_rAtomTable, pChars, eMode,
									  m_nCount ?
									      CntMBXScanner::SCAN_BODY_YES :
									      CntMBXScanner::SCAN_BODY_INIT);
	if (nError == ERRCODE_NONE)
		++m_nCount;
	return nError;
}

inline bool CntMBXBodyScanner::backUp()
{
	if (m_nCount == 0)
		return false;
	bool bSuccessful = m_pScanner->backUpNoEnd();
	if (bSuccessful)
		--m_nCount;
	return bSuccessful;
}

//============================================================================
class CntMBXVersion
{
public:
	enum { NONE = sal_uInt32(-1) };

private:
	sal_uInt32 m_nVersion;

public:
	TYPEINFO();

	CntMBXVersion(sal_uInt32 nTheVersion): m_nVersion(nTheVersion) {}

	virtual ~CntMBXVersion() {}

	sal_uInt32 getVersion() const { return m_nVersion; }
};

//============================================================================
class CntMBXVersion0: public CntMBXVersion
{
public:
	enum MessageType
	{
		TYPE_MESSAGE,
		TYPE_MESSAGEBOX,
		TYPE_NEWSBOX
	};

private:
	MessageType m_eType;

public:
	TYPEINFO();

	CntMBXVersion0(MessageType eTheType): CntMBXVersion(0), m_eType(eTheType)
	{}

	MessageType getType() const { return m_eType; }
};

//============================================================================
class CntMBXMIMEParameter
{
	ByteString m_aValue;
	CntMBXAtom const & m_rAttribute;

public:
	CntMBXMIMEParameter(CntMBXAtom const & rTheAttribute,
						ByteString const & rTheValue):
		m_rAttribute(rTheAttribute), m_aValue(rTheValue) {}

	CntMBXAtom const & getAttribute() const { return m_rAttribute; }

	ByteString getValue() const { return m_aValue; }
};

//============================================================================
DECLARE_LIST(CntMBXMIMEParameterList, CntMBXMIMEParameter *)

//============================================================================
class CntMBXMIMEContentType
{
	CntMBXMIMEParameterList m_aParameters;
	CntMBXAtom const * m_pType;
	CntMBXAtom const * m_pSubtype;

public:
	CntMBXMIMEContentType(): m_pType(0), m_pSubtype(0) {}

	inline ~CntMBXMIMEContentType();

	void setType(CntMBXAtom const * pTheType) { m_pType = pTheType; }

	void setSubtype(CntMBXAtom const * pTheSubtype)
	{ m_pSubtype = pTheSubtype; }

	inline void addParameter(CntMBXAtom const & rAttribute,
							 ByteString const & rValue);

	CntMBXAtom const * getType() const { return m_pType; }

	CntMBXAtom const * getSubtype() const { return m_pSubtype; }

	sal_uInt32 getParameterCount() const { return m_aParameters.Count(); }

	CntMBXMIMEParameter const & getParameter(sal_uInt32 nIndex) const
	{ return *m_aParameters.GetObject(nIndex); }
};

inline CntMBXMIMEContentType::~CntMBXMIMEContentType()
{
	while (m_aParameters.Count())
		delete m_aParameters.Remove(m_aParameters.Count() - 1);
}

inline void CntMBXMIMEContentType::addParameter(CntMBXAtom const & rAttribute,
												ByteString const & rValue)
{
	m_aParameters.Insert(new CntMBXMIMEParameter(rAttribute, rValue),
						 LIST_APPEND);
}

//============================================================================
namespace CntMBXFormat {

ByteString translateWord(ByteString const & rWord);

inline ByteString translateWord(UniString const & rWord)
{ return translateWord(ByteString(rWord, RTL_TEXTENCODING_UTF8)); }

UniString translateXref(UniString const & rHost,
						CntItemListItem const & rList);

inline UniString translateXPriority(CntPriority ePriority)
{ return UniString::CreateFromInt32(ePriority); }

inline ByteString translateBoolean(bool bValue)
{ return bValue ? "yes" : "no"; }

ByteString translateRanges(CntRangesItem const & rRanges);

ByteString translateRFC822AddrSpec(UniString const & rLocalPart,
								   UniString const & rDomain);

bool parseNumericFieldBody(ByteString const & rBody, bool bHex,
						   bool bLeadingZeroes, sal_uInt32 & rValue);

bool parseNumericFieldBody(UniString const & rBody, bool bHex,
						   bool bLeadingZeroes, sal_uInt32 & rValue);

bool parseDateTimeFieldBody(ByteString const & rBody, DateTime & rDateTime);

bool parseDateTimeFieldBody(UniString const & rBody, DateTime & rDateTime);

void decomposeDomainAndPort(UniString const & rDomainAndPort,
							UniString & rDomain, USHORT & rPort);

UniString getLocalServerName(UniString const & rDomainAndPort);

}

} // namespace chaos

#endif // _CHAOS_MBXFORMT_HXX

