// Components for manipulating sequences of characters -*- C++ -*-

// Copyright (C) 1997-1999 Cygnus Solutions
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING.  If not, write to the Free
// Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.

// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.

//
// ISO C++ 14882: 21  Strings library
//

// This file is included by <string>.  It is not meant to be included
// separately.

// Written by Jason Merrill based upon the specification by Takanori Adachi
// in ANSI X3J16/94-0013R2.  Rewritten by Nathan Myers to ISO-14882.

// XXX Iterators should be class objects, for user type-safety.  Current 
//   code uses them interchangeably with pointers, which is pure evil.

#ifndef _CPP_BITS_STRING_TCC
#define _CPP_BITS_STRING_TCC 1

namespace std
{

    // Constructors

// Note: this is the special case for Input Iterators.
//  Input Iterators have a cost structure very different
//  from pointers, calling for a different coding style.

template<class _CharT, class _Traits, class _Alloc>
  template <class _InIter>
     _CharT*
     basic_string<_CharT,_Traits,_Alloc>::
       _S_construct (_InIter __first, _InIter __last, const _Alloc& __a,
		     input_iterator_tag)
{
  if (__first == __last && __a == _Alloc ())
    return _S_empty_rep ()._M_refcopy ();
  // avoid reallocation for common case.
  _CharT __buf[100];
  size_type __i = 0;
  while (__first != __last && __i < sizeof(__buf)/sizeof(_CharT))
    { __buf[__i++] = *__first; ++__first; }
  _Rep* __r = _Rep::_S_create (__i, __a);
  _Traits::copy (__r->_M_data (), __buf, __i);
  __r->_M_length = __i;
  try
    {
      // note: this loop looks precisely this way because
      //   it avoids comparing __first != __last any more
      //   than strictly necessary; != might be expensive!
      for (;;)
	{
	  _CharT* __p = __r->_M_data () + __r->_M_length;
	  _CharT* __end = __r->_M_data () + __r->_M_capacity;
	  for (;;)
	    {
	      if (__first == __last)
		{
		  __r->_M_length = __p - __r->_M_data ();
		  *__p = _CharT();       // grrr.
/**/              return __r->_M_data ();
		}
	      if (__p == __end)
		break;
	      *__p++ = *__first; ++__first;
	    }
	  // assert (__first != __last && __p == __end); allocate more space.
	  size_type __len = __p - __r->_M_data ();
	  _Rep* __another = _Rep::_S_create (__len+1, __a);
	  _Traits::copy (__another->_M_data (), __r->_M_data (), __len);
	  __r->_M_destroy (__a);
	  __r = __another;
	  __r->_M_length = __len;
	}
    }
  catch (...)
    {
      __r->_M_destroy(__a); throw;
    }
  // we never get here.
  return 0;
}

template<class _CharT, class _Traits, class _Alloc>
  template <class _InIter>
     _CharT*
     basic_string<_CharT,_Traits,_Alloc>::
       _S_construct (_InIter __first, _InIter __last, const _Alloc& __a,
		     forward_iterator_tag)
{
  if (__first == __last && __a == _Alloc ())
    return _S_empty_rep ()._M_refcopy ();
  typename iterator_traits<_InIter>::difference_type
    __n = distance(__first, __last);
  __LENGTHERROR(size_type(__n) >= npos);
  _Rep* __r = _Rep::_S_create(__n, __a);
  try { _S_copy_chars (__r->_M_data (), __first, __last); }
  catch (...) { __r->_M_destroy(__a); throw; }
  __r->_M_length = __n;
  __r->_M_data ()[__n] = _CharT();  // grrr.
  return __r->_M_data ();
}

template<class _CharT, class _Traits, class _Alloc>
  _CharT*
  basic_string<_CharT,_Traits,_Alloc>::
    _S_construct (size_type __n, _CharT __c, const _Alloc& __a)
{
  __LENGTHERROR(__n >= npos);
  if (__n == 0 && __a == _Alloc ())
      return _S_empty_rep ()._M_refcopy ();
  _Rep* __r = _Rep::_S_create (__n, __a);
  try { if (__n) _Traits::assign (__r->_M_data (), __n, __c); }
  catch (...) { __r->_M_destroy(__a); throw; }
  __r->_M_length = __n;
  __r->_M_data ()[__n] = _CharT();  // grrr
  return __r->_M_data ();
}

// string constructors:

// inline basic_string<>::basic_string(), in basic_string.h

// copy ctor
template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>::
    basic_string (const basic_string& __str)
  : _M_dataplus (__str._M_rep ()->_M_grab(_Alloc (), __str.get_allocator ()),
		 __str.get_allocator ())
  { }

template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>::
    basic_string (const _Alloc& __a)
  : _M_dataplus (_S_construct (size_type(),_CharT(),__a), __a)
  { }

template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>::basic_string
      (const basic_string& __str, size_type __pos, size_type __n)
  : _M_dataplus (_S_construct (__str._M_check (__pos),
			       __str._M_fold (__pos, __n), _Alloc ()),
		 _Alloc ())
  { }

template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>::basic_string
      (const basic_string& __str, size_type __pos,
       size_type __n, const _Alloc& __a)
  : _M_dataplus (_S_construct (__str._M_check (__pos),
			       __str._M_fold (__pos, __n), __a), __a)
  { }

template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>::basic_string
      (const _CharT* __s, size_type __n, const _Alloc& __a)
  : _M_dataplus (_S_construct (__s, __s+__n, __a), __a)
  { }

template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>::basic_string
      (const _CharT* __s, const _Alloc& __a)
  : _M_dataplus (_S_construct (__s, __s+_Traits::length(__s), __a), __a)
  { }

template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>::basic_string
      (size_type __n, _CharT __c, const _Alloc& __a)
  : _M_dataplus (_S_construct (__n, __c, __a), __a)
  { }

template<class _CharT, class _Traits, class _Alloc>
  template<class _InputIterator>
      basic_string<_CharT,_Traits,_Alloc>::basic_string (
        _InputIterator __begin, _InputIterator __end,
 	const _Alloc& __a)
      : _M_dataplus (_S_construct(__begin, __end, __a), __a)
      { }


  // Modifiers
template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>&
  basic_string<_CharT,_Traits,_Alloc>::
  assign (const basic_string& __str)
  {
    if (_M_rep () != __str._M_rep ())
      {
	// XXX MT
	_CharT* __tmp = __str._M_rep ()->
	  _M_grab (this->get_allocator (), __str.get_allocator ());
	_M_rep ()->_M_dispose (this->get_allocator ());
	_M_data (__tmp);
      }
    return *this;
  }

#if 0  /* XXX this will be used for MT where tstdecr is slow. */
template<class _CharT, class _Traits, class _Alloc>
  void
  basic_string<_CharT,_Traits,_Alloc>::_Rep::
    _M_unlink (const _Alloc& __a) throw ()
{
  // this checks again (even though _M_dispose () just did) because
  //   another thread might have unlinked since the last (unlocked) check.
  if (--_M_state < 0)  // XXX MT
    _M_destroy (__a);
}
#endif

template<class _CharT, class _Traits, class _Alloc>
  void
  basic_string<_CharT,_Traits,_Alloc>::_Rep::
    _M_destroy (const _Alloc& __a) throw ()
{
  _Raw_bytes_alloc(__a).deallocate ((char*)this,
    sizeof (_Rep) + (_M_capacity + 1) * sizeof (_CharT));
}

template<class _CharT, class _Traits, class _Alloc>
  void
  basic_string<_CharT,_Traits,_Alloc>::_M_leak_hard ()
{
  if (_M_rep ()->_M_state > 0) 
    _M_mutate (this->size (), this->size (), 0);
  _M_rep ()->_M_state = -1;
}

template<class _CharT, class _Traits, class _Alloc>
  void
  basic_string<_CharT,_Traits,_Alloc>::_M_mutate(size_type __nsize,
						 size_type __keep,
						 size_type __keep_end)
{
  size_type __old_size = this->size ();
  if (_M_rep ()->_M_state <= 0 && __nsize <= __old_size)
    {
      // work in-place
      _M_rep ()->_M_state = 0;
      if (__keep_end && __nsize < __old_size)
	_Traits::move(_M_data () + __nsize - __keep_end,
		      _M_data () + __old_size - __keep_end, __keep_end);
    }
  else
    {
      _Rep* __r = _Rep::_S_create (__old_size, get_allocator ());
      try
	{
	  if (__keep)
	    _Traits::copy (__r->_M_data (), _M_data (), __keep);
	  if (__keep_end)
	    _Traits::copy (__r->_M_data () + __nsize - __keep_end,
			   _M_data () + __old_size - __keep_end,
			   __keep_end);
	}
      catch (...)
	{ __r->_M_dispose (get_allocator ()); throw; }
      _M_rep ()->_M_dispose (get_allocator ());
      _M_data (__r->_M_data ());
    }
  _M_rep ()->_M_length = __nsize;
  (_M_data ())[__nsize] = _CharT(); // grrr. (per 21.3.4)
     // (you cannot leave those LWG people alone for a second.)
}

template<class _CharT, class _Traits, class _Alloc>
void
basic_string<_CharT,_Traits,_Alloc>::reserve (size_type __res_arg)
{
  if (__res_arg > this->capacity ())
    _M_mutate(__res_arg, this->size (), 0);
}

template<class _CharT, class _Traits, class _Alloc>
  void basic_string<_CharT,_Traits,_Alloc>::swap (basic_string& __s)
{
  if (_M_rep ()->_M_state < 0) _M_rep ()->_M_state = 0;
  if (__s._M_rep ()->_M_state < 0) __s._M_rep ()->_M_state = 0;
  if (get_allocator () == __s.get_allocator ())
    {
      _CharT* __tmp = _M_data ();
      _M_data(__s._M_data ());
      __s._M_data(__tmp);
    }
  else // the code below can usually be optimized away.
    {
      basic_string __tmp1 (_M_ibegin (), _M_iend (), __s.get_allocator ());
      basic_string __tmp2 (__s._M_ibegin (), __s._M_iend (), get_allocator ());
      *this = __tmp2;
      __s = __tmp1;
    }
}

// Memory allocation control.
#ifdef _G_ALLOC_CONTROL
template<class _CharT, class _Traits, class _Alloc>
  bool (*
  basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_excess_slop) (size_t, size_t) =
    basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_default_excess;

template<class _CharT, class _Traits, class _Alloc>
  size_t (*
  basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_frob_size) (size_t) =
   basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_default_frob;
#endif


template<class _CharT, class _Traits, class _Alloc>
inline size_t
#if _G_ALLOC_CONTROL
basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_default_frob (size_t __s)
#else
basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_frob_size (size_t __s)
#endif
{
  // XXX to grow by 3/2 would be better, but non-conforming (for now).
  //     it is tricky but possible to do i*3/2 without overflowing.
  if (__s >= _S_max_size / 2)
    return __s;  // this is huge, no point in leaving room to grow.
  size_t __i = 16; 
  while (__i < __s) __i *= 2;
  return __i;
}

template<class _CharT, class _Traits, class _Alloc>
  basic_string<_CharT,_Traits,_Alloc>::_Rep*
  basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_create (size_t __capacity,
					 	        const _Alloc& __alloc)
{
  __capacity = _S_frob_size (__capacity);
  // might throw, but no worry about a leak: _Rep() does not throw.
  _Rep *__p = new ((void*)_Raw_bytes_alloc(__alloc).allocate (sizeof (_Rep) +
			    (__capacity+1) * sizeof (_CharT))) _Rep;
  __p->_M_capacity = __capacity;
  __p->_M_state = 0;  // one reference
  __p->_M_length = 0;
  return __p;
}


template<class _CharT, class _Traits, class _Alloc>
_CharT*
basic_string<_CharT,_Traits,_Alloc>::_Rep::_M_clone (const _Alloc& __alloc)
{
  _Rep* __p = _Rep::_S_create (_M_length, __alloc);
  if (_M_length)
    {
      try { _Traits::copy(__p->_M_data (), _M_data (), _M_length); }
      catch (...)  { __p->_M_destroy(__alloc); throw; }
    }
  __p->_M_length = _M_length;
  return __p->_M_data ();
}


template<class _CharT, class _Traits, class _Alloc>
inline bool
#ifdef _G_ALLOC_CONTROL
basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_default_excess (size_t __s,
							       size_t __r)
#else
basic_string<_CharT,_Traits,_Alloc>::_Rep::_S_excess_slop (size_t __s,
                                                            size_t __r)
#endif
{
  return 2 * (__s <= 16 ? 16 : __s) < __r;
}


// Static data.

// linker sets _S_empty_rep_storage to all 0s (one reference, empty string)
//   at static init time (before static ctors are run).
template<class _CharT, class _Traits, class _Alloc>
  typename _Alloc::size_type
  basic_string<_CharT,_Traits,_Alloc>::_S_empty_rep_storage[
    (sizeof(_Rep)+sizeof(_CharT)+sizeof(size_type)-1)/sizeof(size_type)];

#if 0  // must be defined in the class
template<class _CharT, class _Traits, class _Alloc>
  const basic_string<_CharT,_Traits,_Alloc>::size_type
  basic_string<_CharT,_Traits,_Alloc>::npos = -1;
#endif

// Capacity:
template<class _CharT, class _Traits, class _Alloc>
void
basic_string<_CharT,_Traits,_Alloc>::resize (size_type __n, _CharT __c)
{
  __LENGTHERROR (__n > max_size ());
  size_type __size = this->size ();
  if (__size < __n)
    this->append(__n - __size, __c);
  else if (__n < __size)
    this->erase(__n);
  // else nothing (in particular, avoid calling _M_mutate() unnecessarily.)
}

// Modifiers.
template<class _CharT, class _Traits, class _Alloc>
  template<class _InputIterator>
    basic_string<_CharT,_Traits,_Alloc>&
  basic_string<_CharT,_Traits,_Alloc>::_M_replace (
    iterator __i1, iterator __i2, _InputIterator __j1, _InputIterator __j2,
    input_iterator_tag)
{
  basic_string __s(__j1, __j2);
  return this->replace(__i1, __i2, __s._M_ibegin (), __s._M_iend ());
}

template<class _CharT, class _Traits, class _Alloc>
  template<class _ForwardIterator>
    basic_string<_CharT,_Traits,_Alloc>&
  basic_string<_CharT,_Traits,_Alloc>::_M_replace (
    iterator __i1, iterator __i2, _ForwardIterator __j1, _ForwardIterator __j2,
    forward_iterator_tag)
{
  size_type __n1 = __i2 - __i1;
  size_type __off1 = __i1 - _M_ibegin ();
  typename iterator_traits<_ForwardIterator>::
    difference_type __n2 = distance(__j1, __j2);
  __LENGTHERROR (max_size () - (this->size () - __n1) <= size_type(__n2));
  size_type __newlen = (this->size () - __n1) + __n2;
  _M_mutate (__newlen, __off1, _M_iend () - __i2);
  // invalidated __i1, __i2
  if (__n2)
    _S_copy_chars(_M_ibegin () + __off1, __j1, __j2);
  return *this;
}

template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>&
basic_string<_CharT,_Traits,_Alloc>::replace (size_type __pos1, size_type __n1,
					      const basic_string& __str,
					      size_type __pos2, size_type __n2)
{
  return this->replace(_M_check (__pos1), _M_fold (__pos1,__n1),
		       __str._M_check(__pos2), __str._M_fold (__pos2,__n2));
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>&
basic_string<_CharT,_Traits,_Alloc>::replace (iterator __i1, iterator __i2,
					      size_type __n2, _CharT __c)
{
  size_type __n1 = __i2 - __i1;
  size_type __off1 = __i1 - _M_ibegin ();
  __LENGTHERROR (max_size () - (this->size () - __n1) <= __n2);
  size_type __newlen = (this->size () - __n1) + __n2;
  _M_mutate (__newlen, __off1, _M_iend () - __i2);
  // invalidated __i1, __i2
  if (__n2)
    _Traits::assign(_M_ibegin () + __off1, __n2, __c);
  return *this;
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::copy (_CharT* __s, size_type __n,
					      size_type __pos) const
{
  __OUTOFRANGE (__pos > length ());

  if (__n > length () - __pos)
    __n = length () - __pos;

  _Traits::copy (__s, data () + __pos, __n);
  // 21.3.5.7 par 3: do not append null.  (good.)
  return __n;
}


// String operations

// XXX note this could be specialized for the standard char_traits
//   specializations to use the same optimizations as strchr.

template<class _CharT, class _Traits, class _Alloc>
const _CharT*
basic_string<_CharT,_Traits,_Alloc>::_S_find (const _CharT* __ptr1,
                                              const _CharT* __ptr2,
                                              _CharT __c)
{
  return find_if(__ptr1, __ptr2, _Char_traits_match<_CharT,_Traits>(__c));
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::find (const _CharT* __s, size_type __pos,
					      size_type __n) const
{
  size_t __xpos = __pos;
  for (; __xpos + __n <= length (); ++__xpos)
    if (_Traits::eq (data () [__xpos], *__s)
        && _Traits::compare (data () + __xpos, __s, __n) == 0)
      return __xpos;
  return npos;
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::find (_CharT __c, size_type __pos) const
{
  if (__pos < this->size())
    {
      const _CharT* __p = _S_find(data () + __pos, _M_iend (), __c);
      if (__p != _M_iend ())
        return __p - data ();
    }
  return npos;
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::rfind (const _CharT* __s, size_type __pos,
					    size_type __n) const
{
  if (__n > length ())
    return npos;

  size_t __xpos = length () - __n;
  if (__xpos > __pos)
    __xpos = __pos;

  for (++__xpos; __xpos-- > 0; )
    if (_Traits::eq (data () [__xpos], *__s)
        && _Traits::compare (data () + __xpos, __s, __n) == 0)
      return __xpos;
  return npos;
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::rfind (_CharT __c, size_type __pos) const
{
  if (1 > length ())
    return npos;

  size_t __xpos = length () - 1;
  if (__xpos > __pos)
    __xpos = __pos;

  for (++__xpos; __xpos-- > 0; )
    if (_Traits::eq (data () [__xpos], __c))
      return __xpos;
  return npos;
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::find_first_of (const _CharT* __s,
						    size_type __pos,
						    size_type __n) const
{
  for (; __pos < length (); ++__pos)
    if (_S_find(__s, __s+__n, data () [__pos]) != __s + __n)
      return __pos;
  return npos;
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::find_last_of (const _CharT* __s,
						   size_type __pos,
						   size_type __n) const
{
  size_type __xpos = this->size();
  if (__xpos == 0) return this->npos;
  if (--__xpos > __pos) __xpos = __pos;
  do
    {
      if (_S_find(__s, __s+__n, data () [__xpos]) != __s + __n)
        return __xpos;
    } while (__xpos-- != 0);
  return npos;
}

template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::find_first_not_of (const _CharT* __s,
							size_type __pos,
							size_type __n) const
{
  size_t __xpos = __pos;
  for (; __xpos < length (); ++__xpos)
    if (_S_find(__s, __s+__n, data () [__xpos]) == __s + __n)
      return __xpos;
  return npos;
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::find_first_not_of (_CharT __c,
							size_type __pos) const
{
  size_t __xpos = __pos;
  for (; __xpos < length (); ++__xpos)
    if (!_Traits::eq (data () [__xpos], __c))
      return __xpos;
  return npos;
}

template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::find_last_not_of (const _CharT* __s,
						       size_type __pos,
						       size_type __n) const
{
  size_type __xpos = this->size();
  if (__xpos == 0) return this->npos;
  if (--__xpos > __pos) __xpos = __pos;
  do
    {
      if (_S_find(__s, __s + __n, data () [__xpos]) == __s + __n)
        return __xpos;
    } while (__xpos--);
  return npos;
}


template<class _CharT, class _Traits, class _Alloc>
basic_string<_CharT,_Traits,_Alloc>::size_type
basic_string<_CharT,_Traits,_Alloc>::find_last_not_of (_CharT __c,
						       size_type __pos) const
{
  size_type __xpos = this->size();
  if (__xpos == 0) return this->npos;
  if (--__xpos > __pos) __xpos = __pos;
  do
    {
      if (!_Traits::eq (data () [__xpos], __c))
        return __xpos;
    } while (__xpos--);
  return npos;
}

template<class _CharT, class _Traits, class _Alloc>
int
basic_string<_CharT,_Traits,_Alloc>::compare (const basic_string& __str) const
{
  size_type __rlen = min (length (), __str.length ());

  int __r = _Traits::compare (data (), __str.data (), __rlen);
  if (__r != 0)
    return __r;
  return length () - __str.length ();
}


template<class _CharT, class _Traits, class _Alloc>
int
basic_string<_CharT,_Traits,_Alloc>::compare (size_type __pos1, size_type __n1,
			                      const basic_string& __str) const
{
  __OUTOFRANGE (__pos1 > length ());

  size_type __rlen = length () - __pos1;
  if (__rlen > __n1)
    __rlen = __n1;
  if (__rlen > __str.length ())
    __rlen = __str.length ();
  int __r = _Traits::compare (data () + __pos1, __str.data (), __rlen);
  if (__r != 0)
    return __r;
  if (__rlen == __n1)
    return 0;
  return (length () - __pos1) - __str.length ();
}


template<class _CharT, class _Traits, class _Alloc>
int
basic_string<_CharT,_Traits,_Alloc>::compare (size_type __pos1, size_type __n1,
					      const basic_string& __str,
					      size_type __pos2,
					      size_type __n2) const
{
  __OUTOFRANGE (__pos1 > length ());
  __OUTOFRANGE (__pos2 > __str.length ());

  const size_type __elen1 = min (length () - __pos1, __n1);
  const size_type __elen2 = min (__str.length () - __pos2, __n2);

  const size_t __rlen = min (__elen1, __elen2);
  int __r = _Traits::compare (data () + __pos1, __str.data () + __pos2, __rlen);
  if (__r != 0)
    return __r;
  return __elen1 - __elen2;
}


template<class _CharT, class _Traits, class _Alloc>
int
basic_string<_CharT,_Traits,_Alloc>::compare (const _CharT* __s) const
{
  int __r = _Traits::compare (data (), __s, length ());
  if (__r != 0)
    return __r;
  else
    return size() - _Traits::length (__s);
  return 0;
}


template<class _CharT, class _Traits, class _Alloc>
int
basic_string <_CharT,_Traits,_Alloc>::compare (size_type __pos, size_type __n1,
					       const _CharT* __s,
					       size_type __n2) const
{
  __OUTOFRANGE (__pos > this->length ());

  const size_type __elen1 = min (length () - __pos, __n1);
  const size_type __elen2 = min (_Traits::length (__s), __n2);

  const size_type __rlen = min (__elen1, __elen2);
  int __r = _Traits::compare (data () + __pos, __s, __rlen);
  if (__r != 0)
    return __r;
  return __elen1 - __elen2;
}


// I/O operators.
template<class _CharT, class _Traits, class _Alloc>
basic_istream<_CharT,_Traits>&
operator>> (basic_istream<_CharT, _Traits>& __is,
	    basic_string<_CharT,_Traits,_Alloc>& __s)
{
  typename basic_istream<_CharT,_Traits>::sentry __cerb (__is);
  if (__cerb)
    {
      __s.resize (0);
      basic_streambuf<_CharT,_Traits>* __sb = __is.rdbuf ();
      streamsize __w = __is.width (0);
      ios_base::iostate __error = ios_base::goodbit;
      while (1)
        {
          typename _Traits::int_type __ch = __sb->sgetc ();
          if (_Traits::eq_int_type(__ch,_Traits::eof()))
            { __error = ios_base::eofbit; break; }
          else if (isspace (__ch)) // XXX wrong
            { break; }
          __s += _Traits::to_char_type(__ch);
          __sb->sbumpc ();
          if (--__w == 0)
            break;
        }

      if (__s.length () == 0) __error |= ios_base::failbit;
      if (__error)
        __is.setstate (__error);  // may throw
    }

  return __is;
}


template<class _CharT, class _Traits, class _Alloc>
basic_ostream<_CharT,_Traits>&
operator<< (basic_ostream<_CharT,_Traits>& __o,
	    const basic_string<_CharT,_Traits,_Alloc>& __s)
{
  return __o.write (__s.data (), __s.length ());
}


template <class _CharT, class _Traits, class _Alloc>
basic_istream<_CharT,_Traits>&
getline (basic_istream<_CharT,_Traits>& __is,
	 basic_string<_CharT,_Traits,_Alloc>& __s,
	 _CharT __delim)
{
  typename basic_istream<_CharT,_Traits>::sentry __cerb(__is);
  if (__cerb)
    {
      basic_streambuf<_CharT,_Traits>* __sb = __is.rdbuf ();
      __s.resize (0);
      streamsize __count = 0;
      ios_base::iostate __error = ios_base::goodbit;

      while (1)
        {
          typename _Traits::int_type __ch = __sb->sbumpc ();
          if (_Traits::eq_int_type(__ch, _Traits::eof()))
            { __error = ios_base::eofbit; break; }
          ++__count;
          if (_Traits::eq(_Traits::to_char_type(__ch), __delim))
            break;
          if (__s.length () == __s.npos - 1)
            { __error = ios_base::failbit; break; }
          __s += _Traits::to_char_type(__ch);
        }

      __is._M_gcount = __count;

      if (__count == 0) __error |= ios_base::failbit;
      if (__error != ios_base::goodbit)
	__is.setstate (__error);  // may throw

    }
  return __is;
}


template <class _CharT, class _Traits, class _Alloc>
void
_S_string_copy(const basic_string<_CharT,_Traits,_Alloc>& __s,
               _CharT* __buf, typename _Alloc::size_type __bufsiz)
{
  typename _Alloc::size_type __bytes =
    __s.size () > __bufsiz-1 ? __bufsiz-1 : __s.size ();
  _Traits::copy(__buf, __s.data (), __bytes);
  __buf[__bytes] = _CharT();
}

} // std::

#endif /* _CPP_BITS_STRING_TCC */

// Local Variables:
// mode:c++
// End:
